<?php

class OneDriveClient
{
    private $tenantId;
    private $clientId;
    private $clientSecret;
    private $driveId;
    private $logFile;
    private $tokenCache;

    public function __construct($tenantId, $clientId, $clientSecret, $driveId)
    {
        $this->tenantId     = $tenantId;
        $this->clientId     = $clientId;
        $this->clientSecret = $clientSecret;
        $this->driveId      = $driveId;

        $this->tokenCache   = __DIR__ . "/token_cache.json";
        $this->logFile      = __DIR__ . "/onedrive_log.txt";
    }

    /******************************************************
     * LOG
     ******************************************************/
    private function log($type, $msg, $context = [])
    {
        $date = date("Y-m-d H:i:s");
        $line = "[$date] [$type] $msg";
        if (!empty($context)) {
            $line .= " | " . json_encode($context, JSON_UNESCAPED_UNICODE);
        }
        file_put_contents($this->logFile, $line . "\n", FILE_APPEND | LOCK_EX);
    }

    /******************************************************
     * TOKEN CON CACHE
     ******************************************************/
    private function getAccessToken()
    {
        if (file_exists($this->tokenCache)) {
            $cache = json_decode(file_get_contents($this->tokenCache), true);

            if (isset($cache["access_token"], $cache["expires_on"]) &&
                $cache["expires_on"] > time()
            ) {
                return $cache["access_token"];
            }
        }

        $url = "https://login.microsoftonline.com/{$this->tenantId}/oauth2/v2.0/token";

        $post = [
            "grant_type"    => "client_credentials",
            "client_id"     => $this->clientId,
            "client_secret" => $this->clientSecret,
            "scope"         => "https://graph.microsoft.com/.default"
        ];

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($post));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        $raw   = curl_exec($ch);
        $error = curl_error($ch);
        $http  = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        curl_close($ch);

        if ($error || $http >= 300) {
            $this->log("ERROR", "Error obteniendo token", [
                "http"  => $http,
                "error" => $error,
                "raw"   => $raw
            ]);
            die("Error obteniendo token");
        }

        $data = json_decode($raw, true);

        file_put_contents($this->tokenCache, json_encode([
            "access_token" => $data["access_token"],
            "expires_on"   => time() + $data["expires_in"] - 60
        ]));

        return $data["access_token"];
    }

    /******************************************************
     * REQUEST A GRAPH
     ******************************************************/
    private function request($method, $url, $body = null, $headers = [])
    {
        $defaultHeaders = [
            "Authorization: Bearer " . $this->getAccessToken()
        ];

        if ($body !== null) {
            $defaultHeaders[] = "Content-Type: application/json";
        }

        $headers = array_merge($defaultHeaders, $headers);

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

        if ($body !== null) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
        }

        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        $raw   = curl_exec($ch);
        $error = curl_error($ch);
        $http  = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        curl_close($ch);

        if ($error || $http >= 400) {
            $this->log("ERROR", "Graph request error", [
                "method" => $method,
                "url"    => $url,
                "http"   => $http,
                "error"  => $error,
                "raw"    => $raw
            ]);
        }

        if ($raw === "" || $http == 204) return [];

        return json_decode($raw, true);
    }

    /******************************************************
     * OBTENER ITEM
     ******************************************************/
    public function getItem($id)
    {
        return $this->request(
            "GET",
            "https://graph.microsoft.com/v1.0/drives/{$this->driveId}/items/{$id}"
        );
    }

    /******************************************************
     * LISTAR CONTENIDO
     ******************************************************/
    public function listChildren($folderId)
    {
        $res = $this->request(
            "GET",
            "https://graph.microsoft.com/v1.0/drives/{$this->driveId}/items/{$folderId}/children"
        );

        return $res["value"] ?? [];
    }

    /******************************************************
     * BREADCRUMB ROBUSTO
     ******************************************************/
    public function buildBreadcrumb($currentId, $rootId)
    {
        $crumbs = [];
        $cursor = $currentId;

        while ($cursor) {
            $item = $this->getItem($cursor);
            if (!$item) break;

            $crumbs[] = ["id" => $item["id"], "name" => $item["name"]];

            if ($item["id"] === $rootId) break;

            if (!isset($item["parentReference"]["id"])) break;

            $cursor = $item["parentReference"]["id"];
        }

        return array_reverse($crumbs);
    }

    /******************************************************
     * SUBIR ARCHIVO
     ******************************************************/
    public function upload($folderId, $localPath, $originalName)
    {
        $url = "https://graph.microsoft.com/v1.0/drives/{$this->driveId}"
             . "/items/{$folderId}:/" . rawurlencode($originalName) . ":/content";

        $data = file_get_contents($localPath);

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "PUT");
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            "Authorization: Bearer " . $this->getAccessToken(),
            "Content-Type: application/octet-stream"
        ]);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        $raw = curl_exec($ch);
        $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($http >= 400) return false;

        return json_decode($raw, true);
    }

    /******************************************************
     * CREAR CARPETA
     ******************************************************/
    public function createFolder($parentId, $name)
    {
        return $this->request(
            "POST",
            "https://graph.microsoft.com/v1.0/drives/{$this->driveId}/items/{$parentId}/children",
            [
                "name" => $name,
                "folder" => new stdClass(),
                "@microsoft.graph.conflictBehavior" => "rename"
            ]
        );
    }

    /******************************************************
     * RENOMBRAR
     ******************************************************/
    public function rename($itemId, $newName)
    {
        return $this->request(
            "PATCH",
            "https://graph.microsoft.com/v1.0/drives/{$this->driveId}/items/{$itemId}",
            ["name" => $newName]
        );
    }

    /******************************************************
     * BORRAR (ARCHIVO O CARPETA)
     ******************************************************/
    public function delete($itemId)
    {
        return $this->request(
            "DELETE",
            "https://graph.microsoft.com/v1.0/drives/{$this->driveId}/items/{$itemId}"
        );
    }
}
